# Flask Login

## Basic usage

The most important part of an application that uses Flask-Login is the
LoginManager class.

``` python
login_manager = LoginManager()
...
login_manager.init_app(app)
```

You will need to provide a user_loader callback which is used to reload
the user object from the user ID stored in the session. It should take
the unicode ID of a user, and return the corresponding user object. For
example:

``` python
@login_manager.user_loader
def load_user(userid):
    return User.get(userid)
```

It should return None (not raise an exception) if the ID is not valid.
(In that case, the ID will manually be removed from the session and
processing will continue.) Once a user has authenticated, you log them
in with the `login_user` function. For example:

``` python
@app.route('/login', methods=['GET', 'POST'])
def login():
    # Here we use a class of some kind to represent and validate our
    # client-side form data. For example, WTForms is a library that will
    # handle this for us.
    form = LoginForm()
    if form.validate_on_submit():
        # Login and validate the user.
        login_user(user)

        flask.flash('Logged in successfully.')

        next = flask.request.args.get('next')
        if not next_is_valid(next):
            return flask.abort(400)

        return flask.redirect(next or flask.url_for('index'))
    return flask.render_template('login.html', form=form)
```

The `login_user` function logs a user in. You should pass the actual
user object to this. If the user's `is_active` property is `False`, they
will not be logged in unless `force` is `True`. This will return `True`
if the log in attempt succeeds, and `False` if it fails.

**Warning**: You MUST validate the value of the next parameter. If you
do not, your application will be vulnerable to open redirects.

It's that simple. You can then access the logged-in user with the
current_user proxy, which is available in every template:

    {% if current_user.is_authenticated() %}
      Hi {{ current_user.name }}!
    {% endif %}

Views that require your users to be logged in can be decorated with the
login_required decorator:

``` python
@app.route("/settings")
@login_required
def settings():
    pass
```

When the user is ready to log out:

``` python
@app.route("/logout")
@login_required
def logout():
    logout_user()
    return redirect(somewhere)
```

### Requirements

The class that you use to represent users needs to implement these
properties and methods:

-   is_authenticated: Returns True if the user is authenticated.
-   is_active: Returns True if this is an active user - in addition to
    being authenticated, they also have activated their account, not
    been suspended, or any condition your application has for rejecting
    an account. Inactive accounts may not log in (without being forced
    of course).
-   is_anonymous: Returns True if this is an anonymous user. (Actual
    users should return False instead.)
-   get_id(): Returns a unicode that uniquely identifies this user, and
    can be used to load the user from the user_loader callback. Note
    that this must be a unicode - if the ID is natively an int or some
    other type, you will need to convert it to unicode.

To make implementing a user class easier, you can inherit from
`UserMixin`, which provides default implementations for all of these
methods. (It's not required, though.)

## Unauthorize

You can restrict a page with `@login_required` decorator. If a user not
logged in try to access he is redirected to a login page. If this page
is not set, a 404 error will happen.

``` python
@app.route('/post')
@login_required
def post():
    pass
```

You can customize this using:

``` python
@login_manager.unauthorized_handler
def unauthorized():
    # do stuff
    return a_response
```

The name of the log in view can be set as LoginManager.login_view. For
example:

``` python
login_manager.login_view = "users.login"
```

You also can define the default message:

``` python
login_manager.login_message = u"Bonvolu ensaluti por uzi tio paĝo."
```

### Authorize from header

Sometimes you want to login users without using cookies, such as using
header values or an api key passed as a query argument. In these cases,
you should use the request_loader callback.

``` python
@login_manager.request_loader
def load_user_from_request(request):

    # first, try to login using the api_key url arg
    api_key = request.args.get('api_key')
    if api_key:
        user = User.query.filter_by(api_key=api_key).first()
        if user:
            return user

    # next, try to login using Basic Auth
    api_key = request.headers.get('Authorization')
    if api_key:
        api_key = api_key.replace('Basic ', '', 1)
        try:
            api_key = base64.b64decode(api_key)
        except TypeError:
            pass
        user = User.query.filter_by(api_key=api_key).first()
        if user:
            return user

    # finally, return None if both methods did not login the user
    return None
```

### Authorize from tokens

You can hash the username and password combined, or something similar.
To add an alternative token, add a method to your user objects.

`get_auth_token()` function returns an authentication token (as unicode)
for the user. The auth token should uniquely identify the user, and
preferably not be guessable. Correspondingly, you should set a
token_loader function on the LoginManager, which takes a token (as
stored in the cookie) and returns the appropriate User object.

## Sessions

### Fresh logins

When a user logs in, their session is marked as "fresh," which indicates
that they actually authenticated on that session. When their session is
destroyed and they are logged back in with a "remember me" cookie, it is
marked as "non-fresh." login_required does not differentiate between
freshness, which is fine for most pages. However, sensitive actions like
changing one's personal information should require a fresh login.

fresh_login_required, in addition to verifying that the user is logged
in, will also ensure that their login is fresh. If not, it will send
them to a page where they can re-enter their credentials. You can
customize its behavior in the same ways as you can customize
login_required, by setting LoginManager.refresh_view,
needs_refresh_message, and needs_refresh_message_category:

``` python
login_manager.refresh_view = "accounts.reauthenticate"
login_manager.needs_refresh_message = (
    u"To protect your account, please reauthenticate to access this page."
)
login_manager.needs_refresh_message_category = "info"
```

Or by providing your own callback to handle refreshing:

``` python
@login_manager.needs_refresh_handler
def refresh():
    # do stuff
    return a_response
```

To mark a session as fresh again, call the confirm_login function.

### Session protection

You can configure session protection on the LoginManager, and in the
app's configuration. If it is enabled, it can operate in either basic or
strong mode. To set it on the LoginManager, set the session_protection
attribute to \"basic\" or \"strong\":

``` python
login_manager.session_protection = "strong"
# Or, to disable it:
login_manager.session_protection = None
```

By default, it is activated in \"basic\" mode. It can be disabled in the
app's configuration by setting the SESSION_PROTECTION setting to None,
\"basic\", or \"strong\".

## Other functionalities

### Anonymous users

When a user is not actually logged in, current_user is set to an
AnonymousUserMixin object. You can use another class with\...

``` python
login_manager.anonymous_user = MyAnonymousUser
```

### Remember me

To implement it just pass `remember=True` to the `login_user` call. A
cookie will be saved on the user's computer, and then Flask-Login will
automatically restore the user ID from that cookie if it is not in the
session. The cookie is tamper-proof, so if the user tampers with it
(i.e. inserts someone else's user ID in place of their own), the cookie
will merely be rejected, as if it was not there.

However, you can (and should, if your application handles any kind of
sensitive data) provide additional infrastructure to increase the
security of your remember cookies.

### Cookies

The details of the cookie can be customized in the application settings.

-   REMEMBER_COOKIE_NAME, The name of the cookie to store the "remember
    me" information in.
-   REMEMBER_COOKIE_DURATION, The amount of time before the cookie
    expires, as a datetime.timedelta object.
-   REMEMBER_COOKIE_DOMAIN, If the "Remember Me" cookie should cross
    domains, set the domain value here (i.e. .example.com would allow
    the cookie to be used on all subdomains of example.com).
-   REMEMBER_COOKIE_PATH, Limits the "Remember Me" cookie to a certain
    path.
