# Flask MongoAlchemy

## Install

    pip install Flask-MongoAlchemy

## Use

### Initialization

``` python
from flask import Flask
from flask.ext.mongoalchemy import MongoAlchemy
app = Flask(__name__)
app.config['MONGOALCHEMY_DATABASE'] = 'library'
db = MongoAlchemy(app)

class Author(db.Document):
    name = db.StringField()

class Book(db.Document):
    title = db.StringField()
    author = db.DocumentField(Author)
    year = db.IntField()
```

There are other ways to initialize it:

``` python
app = Flask(__name__)
db = MongoAlchemy(app)
```

``` python
db = MongoAlchemy()
def init_app():
    app = Flask(__name__)
    db.init_app(app)
    return app
```

### Create, update, delete

Now you can create authors and books:

``` python
>>> from application import Author, Book
>>> mark_pilgrim = Author(name='Mark Pilgrim')
>>> dive = Book(title='Dive Into Python', author=mark_pilgrim, year=2004)
>>> mark_pilgrim.save()
>>> dive.save()
```

If you make any changes on a document, you may call save() again:

``` python
>>> mark_pilgrim.name = 'Mark Stalone'
>>> mark_pilgrim.save()
```

And you can remove a document from the database by calling its remove()
method:

``` python
>>> dive.remove()
```

### Queries

Every document has a query class property. It's very simple to use it:

``` python
>>> mark = Author.query.get('76726')
>>> mark.name = 'Mark Pilgrim'
>>> mark.save()
```

You also can use the filter method instead of the get() method:

``` python
>>> mark = Author.query.filter(Author.name == 'Mark Pilgrim').first()
>>> mark.name = 'Steve Jobs'
>>> mark.save()
```

#### Advanced

You can create your own queries:

``` python
from flask.ext.mongoalchemy import BaseQuery
from application import db

class MyCustomizedQuery(BaseQuery):

    def get_johns(self):
        return self.filter(self.type.first_name == 'John')

class Person(db.Document):
    query_class = MyCustomizedQuery
    name = db.StringField()
```

There are other functions:

-   first_or_404()
-   get(mongo_id)
-   get_or_404(mongo_id)

#### Pagination

You can call a method for a query:
`paginate(page, per_page=20, error_out=True)`. This method returns a
`Pagination` object in which you can call\...

-   has_next()
-   has_prev()
-   items = None
-   next(error_out=False)
-   next_num
-   page = None
-   pages
-   per_page = None
-   prev(error_out=False)
-   prev_num
-   query = None
-   total = None

[Documentation for these
methods](https://pythonhosted.org/Flask-MongoAlchemy/index.html#flask.ext.mongoalchemy.Pagination.has_next)

## Configuration

``` python
>>> app.config['MONGOALCHEMY_SERVER_AUTH'] = False
```

There are values to change in Mongo config:

-   MONGOALCHEMY_DATABASE
-   MONGOALCHEMY_SERVER
-   MONGOALCHEMY_PORT
-   MONGOALCHEMY_USER
-   MONGOALCHEMY_PASSWORD
-   MONGOALCHEMY_SAFE_SESSION
-   MONGOALCHEMY_OPTIONS
-   MONGOALCHEMY_SERVER_AUTH
-   MONGOALCHEMY_REPLICA_SET
-   MONGOALCHEMY_CONNECTION_STRING

When MongoAlchemy or init_app() are invoked with only one argument (the
Flask instance), a configuration value prefix of MONGOALCHEMY is
assumed; this can be overridden with the config_prefix argument, for
example:

``` python
app = Flask(__name__)
# defaulting to MONGOENGINE prefix
mongo1 = MongoAlchemy(app)
# using another database config
app.config['OTHER_DBNAME'] = 'other'
mongo2 = MongoAlchemy(app, config_prefix='OTHER')
```
