# Flask MongoEngine

## Basic usage

### Install

    pip install flask-mongoengine

### Setting up the DB

``` python
from flask import Flask
from flask.ext.mongoengine import MongoEngine

app = Flask(__name__)
app.config.from_pyfile('the-config.cfg')
db = MongoEngine(app)
```

Or, if you are setting up your database before your app is
initialized\....

``` python
from flask import Flask
from flask.ext.mongoengine import MongoEngine
db = MongoEngine()
...
app = Flask(__name__)
app.config.from_pyfile('the-config.cfg')
db.init_app(app)
```

### Configuration

``` python
app.config['MONGODB_SETTINGS'] = {
    'db': 'project1',
    'host': '192.168.1.35',
    'port': 12345,
    'username':'webapp',
    'password':'pwd123'
}
```

``` python
app.config['MONGODB_DB'] = 'project1'
app.config['MONGODB_HOST'] = '192.168.1.35'
app.config['MONGODB_PORT'] = 12345
app.config['MONGODB_USERNAME'] = 'webapp'
app.config['MONGODB_PASSWORD'] = 'pwd123'
```

### Model

``` python
from app import db


class Post(db.Document):
    created_at = db.DateTimeField(default=datetime.datetime.now, required=True)
    title = db.StringField(max_length=255, required=True)
    slug = db.StringField(max_length=255, required=True)
    body = db.StringField(required=True)
    comments = db.ListField(db.EmbeddedDocumentField('Comment'))

    def get_absolute_url(self):
        return url_for('post', kwargs={"slug": self.slug})

    def __unicode__(self):
        return self.title

    meta = {
        'allow_inheritance': True,
        'indexes': ['-created_at', 'slug'],
        'ordering': ['-created_at']
    }
```

### Use

Flask MongoEngine attaches methods to MongoEngine:

-   get_or_404: works like .get(), but calls abort(404) if the object
    DoesNotExist.
-   first_or_404: same as above, except for .first().
-   paginate: paginates the QuerySet. Takes two arguments, page and
    per_page.
-   paginate_field: paginates a field from one document in the QuerySet.
    Arguments: field_name, doc_id, page, per_page.

``` python
# 404 if object doesn't exist
def view_todo(todo_id):
    todo = Todo.objects.get_or_404(_id=todo_id)
..

# Paginate through todo
def view_todos(page=1):
    paginated_todos = Todo.objects.paginate(page=page, per_page=10)

# Paginate through tags of todo
def view_todo_tags(todo_id, page=1):
    todo = Todo.objects.get_or_404(_id=todo_id)
    paginated_tags = todo.paginate_field('tags', page, per_page=10)
```

You can use them in templates:

    {% macro render_pagination(pagination, endpoint) %}
      <div class=pagination>
      {%- for page in pagination.iter_pages() %}
        {% if page %}
          {% if page != pagination.page %}
            <a href="{{ url_for(endpoint, page=page) }}">{{ page }}</a>
          {% else %}
            <strong>{{ page }}</strong>
          {% endif %}
        {% else %}
          <span class=ellipsis>…</span>
        {% endif %}
      {%- endfor %}
      </div>
    {% endmacro %}

## Other uses

### Session storage

To use MongoEngine as your session store simple configure the session
interface:

``` python
from flask.ext.mongoengine import MongoEngine, MongoEngineSessionInterface
app = Flask(__name__)
db = MongoEngine(app)
app.session_interface = MongoEngineSessionInterface(db)
```

### Debug Toolbar Panel

If you use the Flask-DebugToolbar you can add
'flask.ext.mongoengine.panels.MongoDebugPanel' to the DEBUG_TB_PANELS
config list and then it will automatically track your queries.

``` python
from flask import Flask from flask_debugtoolbar import DebugToolbarExtension
app = Flask(__name__) 
app.config[‘DEBUG_TB_PANELS’] = [‘flask.ext.mongoengine.panels.MongoDebugPanel’] 
db = MongoEngine(app) 
toolbar = DebugToolbarExtension(app)
```
