//
//  Square.m
//  GLSquare
//
//  Created by alfred visyon on 26/11/15.
//  Copyright © 2015 Secret Lab. All rights reserved.
//

#import "Square.h"
#import "GLProgram.h"
#import <GLKit/GLKit.h>
#import <OpenGLES/ES2/gl.h>
#import <OpenGLES/ES2/glext.h>
#import <UIKit/UIKit.h>
#import <GLKit/GLKit.h>

typedef struct {
    GLKVector3 position; // the location of each vertex in space
    GLKVector2 textureCoordinates; // the texture coordinates for each vertex
} Vertex;

const Vertex SquareVertices[] = {
    {{-1, -1 , 0}, {0,0}}, // bottom left
    {{1, -1 , 0}, {1,0}},  // bottom right
    {{1, 1 , 0}, {1,1}},   // top right
    {{-1, 1 , 0}, {0,1}},  // top left
};

const GLubyte SquareTriangles[] = {
    0, 1, 2, // BL -> BR -> TR
    2, 3, 0  // TR -> TL -> BL
};

const float Vertices[12] = {
    -1.0f, -1.0f, 0.0f,
    1.0f, -1.0f, 0.0f,
    1.0f, 1.0f, 0.0f,
    -1.0f, 1.0f, 0.0f
};

const float VertCoord[8] = {
    0,0,
    1,0,
    1,1,
    0,1
};

const GLubyte VertInd[6] = {
    0, 1, 2, // BL -> BR -> TR
    2, 3, 0  // TR -> TL -> BL
};

@interface Square() {
    GLuint _vertexBuffer; // contains the collection of vertices used to describe position of each corner
    GLuint _indexBuffer;  // indicates which vertices should be used in each triangle used to make up the square
    GLProgram* _program;
    GLint uMVPMatrix;
    GLKMatrix4 _modelViewProjectionMatrix;
    GLuint _textCoordBuffer;
    GLuint _vertexTexCoordAttributeIndex;
    GLuint _vertexAttributeIndex;
    GLuint _samplerIndex;
    GLuint texture;
    GLuint textureCoordId;
    GLuint pospos;
    GLuint tcopos;
    CVOpenGLESTextureRef textureRef;
}
- (GLuint)createTexture2dFromImage:(UIImage*)image;
@end

@implementation Square

-(void) createY:(float) posY Image:(NSString*) imageName {
    _program = [[GLProgram alloc] initWithVertexShader:SIMPLE_VERTEX FragmentShader:SIMPLE_FRAGMENT];
    
    [_program addAttribute:@"position"];
    [_program addAttribute:@"texCoord"];
    _vertexTexCoordAttributeIndex = [_program attributeIndex:@"texCoord"];

    if (![_program link])
    {
        NSString *programLog = [_program programLog];
        NSLog(@"Program link log: %@", programLog);
        NSString *fragmentLog = [_program fragmentShaderLog];
        NSLog(@"Fragment shader compile log: %@", fragmentLog);
        NSString *vertexLog = [_program vertexShaderLog];
        NSLog(@"Vertex shader compile log: %@", vertexLog);
        _program = nil;
        NSAssert(NO, @"Falied to link HalfSpherical shaders");
    }
    
    uMVPMatrix = [_program uniformIndex:@"modelViewProjectionMatrix"];
    _samplerIndex = [_program uniformIndex:@"SamplerY"];
    
    glBindAttribLocation(_program.program, GLKVertexAttribPosition, "position");
    //glBindAttribLocation(_program.program, GLKVertexAttribTexCoord0, "texCoord");
    
    UIImage* image = [UIImage imageNamed:imageName];
    [self createTexture2dFromImage:image];
    
    glGenBuffers(1, &_vertexBuffer);
    glBindBuffer(GL_ARRAY_BUFFER, _vertexBuffer);
    glBufferData(GL_ARRAY_BUFFER, sizeof(SquareVertices), SquareVertices, GL_STATIC_DRAW);
    
    glGenBuffers(1, &_indexBuffer);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _indexBuffer);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(SquareTriangles), SquareTriangles, GL_STATIC_DRAW);
     
     glEnableVertexAttribArray(GLKVertexAttribPosition);
     glVertexAttribPointer(GLKVertexAttribPosition, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex), 0);
     glEnableVertexAttribArray(_vertexTexCoordAttributeIndex);
     glVertexAttribPointer(_vertexTexCoordAttributeIndex, 2, GL_FLOAT, GL_FALSE, sizeof(Vertex), (void*)offsetof(Vertex, textureCoordinates));
    
    float aspectRatio = 0.56;
    float fieldOfViewDegrees = 60.0;
    GLKMatrix4 projectionMatrix = GLKMatrix4MakePerspective(GLKMathDegreesToRadians(fieldOfViewDegrees), aspectRatio, 0.1, 10.0);
    GLKMatrix4 modelViewMatrix = GLKMatrix4MakeTranslation(0.0f, posY, -6.0f);
    _modelViewProjectionMatrix = GLKMatrix4Multiply(projectionMatrix, modelViewMatrix);
    
    glUniform1i(_samplerIndex, texture);
    [_program use];
    glUniform1i(_samplerIndex, texture);
}

-(void) draw {
    glUniformMatrix4fv(uMVPMatrix, 1, 0, _modelViewProjectionMatrix.m);
    
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, texture);
    glUniform1i(_samplerIndex, texture);
    
    
    glEnableVertexAttribArray(GLKVertexAttribPosition);
    glVertexAttribPointer(GLKVertexAttribPosition, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex), 0);
    glEnableVertexAttribArray(_vertexTexCoordAttributeIndex);
    glVertexAttribPointer(_vertexTexCoordAttributeIndex, 2, GL_FLOAT, GL_FALSE, sizeof(Vertex), (void*)offsetof(Vertex, textureCoordinates));
    
    glDrawElements(GL_TRIANGLE_STRIP,6, GL_UNSIGNED_BYTE, 0);
}

- (GLuint) createTexture2dFromImage:(UIImage*) image
{
    int numComponents = 4;
    
    if (image == nil)
        NSLog(@"Do real error checking here");
    
    CGImageRef imageRef = [image CGImage];
    //dividable by 32
    int width = CGImageGetWidth(imageRef) > 256 ? 512 : 256;
    int height = width;//CGImageGetHeight(imageRef) > 64 ? 128 : 64;
    const int BUFF_SIZE = width * height * numComponents;
    
    char* colorData = (char *)malloc(BUFF_SIZE);
    
    CGColorSpaceRef colorSpace = CGColorSpaceCreateDeviceRGB();
    NSUInteger bytesPerPixel = 4;
    NSUInteger bytesPerRow = bytesPerPixel * width;
    NSUInteger bitsPerComponent = 8;
    CGContextRef context = CGBitmapContextCreate(colorData, width, height,
                                                 bitsPerComponent, bytesPerRow, colorSpace,
                                                 kCGBitmapByteOrder32Big | kCGImageAlphaPremultipliedLast );
    NSAssert(context != nil, @"bmp context");
    CGColorSpaceRelease(colorSpace);
    CGContextSetBlendMode(context, kCGBlendModeCopy);
    
    CGContextDrawImage(context, CGRectMake(0, 0, width, height), imageRef);
    CGImageRef cgImage = CGBitmapContextCreateImage(context);
    UIImage *img = [[UIImage alloc]initWithCGImage:cgImage];
    if( img == nil ) NSLog(@"img 1 Error");
    
    CGContextRelease(context);
    glPixelStorei(GL_PACK_ALIGNMENT, 1);
    glEnable(GL_TEXTURE_2D);
    glGenTextures(1, &texture);
    
    NSLog(@"Texture Log = %d", texture);
    glActiveTexture(GL_TEXTURE0 + texture);
    glBindTexture(GL_TEXTURE_2D, texture);
    
    // Setup texture parameters
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);//disable mipmapping
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    //set the bits of the texture
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, colorData);
    
    // We can delete the application copy of the texture data now
    free(colorData); colorData = 0;
    
    return texture;
}

@end
