"""Miscellaneous utility functions

This program is part of "Dive Into Python", a free Python book for
experienced programmers.  Visit http://diveintopython.org/ for the
latest version.
"""

__author__ = "Mark Pilgrim (f8dy@diveintopython.org)"
__version__ = "1.1"
__date__ = "27 August 2001"
__copyright__ = "Copyright (c) 2001 Mark Pilgrim"
__license__ = "Python"

def hardwrap(s, maxcol=72):
    """hard wrap string to maxcol columns

    Example:
    >>> print hardwrap("This is a test of the emergency broadcasting system", 25)
    This is a test of the
    emergency broadcasting
    system.
    """
    import re
    pattern = re.compile(r'.*\s')
    def wrapline(s, pattern=pattern, maxcol=maxcol):
        lines = []
        start = 0
        while 1:
            if len(s) - start <= maxcol: break
            m = pattern.match(s[start:start + maxcol])
            if not m: break
            newline = m.group()
            lines.append(newline)
            start += len(newline)
        lines.append(s[start:])
        return "\n".join([s.rstrip() for s in lines])
    return "\n".join(map(wrapline, s.split("\n")))

def openAnything(source):
    """URI, filename, or string --> stream

    This function lets you define parsers that take any input source
    (URL, pathname to local or network file, or actual data as a string)
    and deal with it in a uniform manner.  Returned object is guaranteed
    to have all the basic stdio read methods (read, readline, readlines).
    Just .close() the object when you're done with it.
    
    Examples:
    >>> from xml.dom import minidom
    >>> sock = openAnything("http://localhost/kant.xml")
    >>> doc = minidom.parse(sock)
    >>> sock.close()
    >>> sock = openAnything("c:\\inetpub\\wwwroot\\kant.xml")
    >>> doc = minidom.parse(sock)
    >>> sock.close()
    >>> sock = openAnything("<ref id='conjunction'><text>and</text><text>or</text></ref>")
    >>> doc = minidom.parse(sock)
    >>> sock.close()
    """

    # try to open with urllib (if source is http, ftp, or file URL)
    import urllib
    try:
        return urllib.urlopen(source)
    except IOError:
        pass
    
    # try to open with native open function (if source is pathname)
    try:
        return open(source)
    except IOError:
        pass
    
    # assume source is string, create stream
    import StringIO
    return StringIO.StringIO(source)
